(function (window, document) {
    'use strict';

    var KEY_LEFT = 37;
    var KEY_RIGHT = 39;
    var KEY_HOME = 36;
    var KEY_END = 35;

    function activateTab(buttons, panels, targetId) {
        buttons.forEach(function (button) {
            var isActive = button.dataset.tfsTabTarget === targetId;
            button.classList.toggle('is-active', isActive);
            button.setAttribute('aria-selected', isActive ? 'true' : 'false');
            button.setAttribute('tabindex', isActive ? '0' : '-1');
        });

        panels.forEach(function (panel) {
            var isActive = panel.id === targetId;
            panel.classList.toggle('is-active', isActive);
            panel.setAttribute('aria-hidden', isActive ? 'false' : 'true');
        });
    }

    function getNextIndex(current, length, direction) {
        if (direction === 'next') {
            return (current + 1) % length;
        }

        if (direction === 'prev') {
            return (current - 1 + length) % length;
        }

        return current;
    }

    function bindKeyboardNavigation(buttons) {
        buttons.forEach(function (button, index) {
            button.addEventListener('keydown', function (event) {
                var key = event.which || event.keyCode;
                var targetIndex = index;

                if (key === KEY_RIGHT) {
                    targetIndex = getNextIndex(index, buttons.length, 'next');
                } else if (key === KEY_LEFT) {
                    targetIndex = getNextIndex(index, buttons.length, 'prev');
                } else if (key === KEY_HOME) {
                    targetIndex = 0;
                } else if (key === KEY_END) {
                    targetIndex = buttons.length - 1;
                } else {
                    return;
                }

                event.preventDefault();
                buttons[targetIndex].focus();
                buttons[targetIndex].click();
            });
        });
    }

    function initTabs(container) {
        if (container.dataset.tfsTabsReady) {
            return;
        }

        var buttons = Array.prototype.slice.call(
            container.querySelectorAll('.tfs-tabs__nav-button')
        );
        var panels = Array.prototype.slice.call(
            container.querySelectorAll('.tfs-tabs__panel')
        );

        if (!buttons.length || !panels.length) {
            return;
        }

        buttons.forEach(function (button) {
            button.addEventListener('click', function () {
                var targetId = button.dataset.tfsTabTarget;
                activateTab(buttons, panels, targetId);
            });
        });

        bindKeyboardNavigation(buttons);

        var activeButton = buttons.find(function (button) {
            return button.classList.contains('is-active');
        }) || buttons[0];

        if (activeButton) {
            activateTab(buttons, panels, activeButton.dataset.tfsTabTarget);
        }

        container.dataset.tfsTabsReady = 'true';
    }

    function init(scope) {
        var context = scope instanceof HTMLElement ? scope : document;
        var containers = context.querySelectorAll('.tfs-tabs');

        containers.forEach(initTabs);
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function () {
            init(document);
        });
    } else {
        init(document);
    }

    if (window.elementorFrontend && window.elementorFrontend.hooks) {
        window.elementorFrontend.hooks.addAction(
            'frontend/element_ready/tfs-shortcode-tabs.default',
            init
        );
    }
})(window, document);
